# Łamanie szyfru Vigenère’a:
# https://www.nostarch.com/crackingcodes/ (na licencji BSD).

import itertools, re
import vigenereCipher, pyperclip, freqAnalysis, detectEnglish

LETTERS = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ'
SILENT_MODE = False  # Wartość True oznacza, że program nie będzie wyświetlał żadnych danych wyjściowych.
NUM_MOST_FREQ_LETTERS = 4  # Dla podklucza będzie sprawdzana podana liczba liter.
MAX_KEY_LENGTH = 16  # Nie będą sprawdzane klucze dłuższe niż podana wartość.
NONLETTERS_PATTERN = re.compile('[^A-Z]')


def main():
    # Ten tekst możesz skopiować i wkleić z pliku kodu źródłowego, który znajdziesz na stronie
    # https://www.nostarch.com/crackingcodes/.
    ciphertext = """Adiz Avtzqeci Tmzubb wsa m Pmilqev halpqavtakuoi, lgouqdaf, kdmktsvmztsl, izr xoexghzr kkusitaaf. Vz wsa twbhdg ubalmmzhdad qz hce vmhsgohuqbo ox kaakulmd gxiwvos, krgdurdny i rcmmstugvtawz ca tzm ocicwxfg jf "stscmilpy" oid "uwydptsbuci" wabt hce Lcdwig eiovdnw. Bgfdny qe kddwtk qjnkqpsmev ba pz tzm roohwz at xoexghzr kkusicw izr vrlqrwxist uboedtuuznum. Pimifo Icmlv Emf DI, Lcdwig owdyzd xwd hce Ywhsmnemzh Xovm mby Cqxtsm Supacg (GUKE) oo Bdmfqclwg Bomk, Tzuhvif'a ocyetzqofifo ositjm. Rcm a lqys ce oie vzav wr Vpt 8, lpq gzclqab mekxabnittq tjr Ymdavn fihog cjgbhvnstkgds. Zm psqikmp o iuejqf jf lmoviiicqg aoj jdsvkavs Uzreiz qdpzmdg, dnutgrdny bts helpar jf lpq pjmtm, mb zlwkffjmwktoiiuix avczqzs ohsb ocplv nuby swbfwigk naf ohw Mzwbms umqcifm. Mtoej bts raj pq kjrcmp oo tzm Zooigvmz Khqauqvl Dincmalwdm, rhwzq vz cjmmhzd gvq ca tzm rwmsl lqgdgfa rcm a kbafzd-hzaumae kaakulmd, hce SKQ. Wi 1948 Tmzubb jgqzsy Msf Zsrmsv'e Qjmhcfwig Dincmalwdm vt Eizqcekbqf Pnadqfnilg, ivzrw pq onsaafsy if bts yenmxckmwvf ca tzm Yoiczmehzr uwydptwze oid tmoohe avfsmekbqr dn eifvzmsbuqvl tqazjgq. Pq kmolm m dvpwz ab ohw ktshiuix pvsaa at hojxtcbefmewn, afl bfzdakfsy okkuzgalqzu xhwuuqvl jmmqoigve gpcz ie hce Tmxcpsgd-Lvvbgbubnkq zqoxtawz, kciup isme xqdgo otaqfqev qz hce 1960k. Bgfdny'a tchokmjivlabk fzsmtfsy if i ofdmavmz krgaqqptawz wi 1952, wzmz vjmgaqlpad iohn wwzq goidt uzgeyix wi tzm Gbdtwl Wwigvwy. Vz aukqdoev bdsvtemzh rilp rshadm tcmmgvqg (xhwuuqvl uiehmalqab) vs sv mzoejvmhdvw ba dmikwz. Hpravs rdev qz 1954, xpsl whsm tow iszkk jqtjrw pug 42id tqdhcdsg, rfjm ugmbddw xawnofqzu. Vn avcizsl lqhzreqzsy tzif vds vmmhc wsa eidcalq; vds ewfvzr svp gjmw wfvzrk jqzdenmp vds vmmhc wsa mqxivmzhvl. Gv 10 Esktwunsm 2009, fgtxcrifo mb Dnlmdbzt uiydviyv, Nfdtaat Dmiem Ywiikbqf Bojlab Wrgez avdw iz cafakuog pmjxwx ahwxcby gv nscadn at ohw Jdwoikp scqejvysit xwd "hce sxboglavs kvy zm ion tjmmhzd." Sa at Haq 2012 i bfdvsbq azmtmd'g widt ion bwnafz tzm Tcpsw wr Zjrva ivdcz eaigd yzmbo Tmzubb a kbmhptgzk dvrvwz wa efiohzd."""
    hackedMessage = hackVigenere(ciphertext)

    if hackedMessage != None:
        print('Deszyfrowana wiadomość została skopiowana do schowka:')
        print(hackedMessage)
        pyperclip.copy(hackedMessage)
    else:
        print('Deszyfrowanie zakończyło się niepowodzeniem.')


def findRepeatSequencesSpacings(message):
    # Analiza ciągu tekstowego w zmiennej message oraz odszukanie wszelkich powtarzających
    # się sekwencji od 3 do 5 liter. Wartością zwrotną jest słownik, którego klucze to znalezione sekwencje, a wartości
    # to listy odstępów (czyli liczby liter między powtarzającymi się sekwencjami).

    # Użycie wyrażenia regularnego do usunięcia z message znaków innych niż litery.
    message = NONLETTERS_PATTERN.sub('', message.upper())

    # Przygotowanie list odstępów między sekwencjami znalezionymi w ciągu tekstowym message.
    seqSpacings = {}  # Klucze to sekwencje, wartościami są listy odstępów w postaci liczb całkowitych.
    for seqLen in range(3, 6):
        for seqStart in range(len(message) - seqLen):
            # Określenie sekwencji i umieszczenie jej w zmiennej seq.
            seq = message[seqStart:seqStart + seqLen]

            # Wyszukanie danej sekwencji w pozostałej części ciągu tekstowego message.
            for i in range(seqStart + seqLen, len(message) - seqLen):
                if message[i:i + seqLen] == seq:
                    # Znaleziono powtarzającą się sekwencję.
                    if seq not in seqSpacings:
                        seqSpacings[seq] = []  # Inicjalizacja pustej listy.

                    # Dołączenie wartości odstępu między sekwencjami
                    # powtórzoną i pierwotną.
                    seqSpacings[seq].append(i - seqStart)
    return seqSpacings


def getUsefulFactors(num):
    # Zwraca listę użytecznych dzielników num; tutaj słowo "użytecznych" oznacza dzielniki
    # mniejsze niż wartość MAX_KEY_LENGTH + 1, np. wywołanie 
    # getUsefulFactors(144) zwraca listę [2, 3, 4, 6, 8, 9, 12, 16].

    if num < 2:
        return []  # Liczby mniejsze niż 2 nie mają zbyt użytecznych dzielników.

    factors = []  # Lista znalezionych dzielników.

    # Podczas wyszukiwania dzielników, trzeba sprawdzić liczby całkowite
    # tylko do wartości wyrażonej w MAX_KEY_LENGTH.
    for i in range(2, MAX_KEY_LENGTH + 1):  # Nie sprawdzamy wartości 1, ponieważ nie jest zbyt użyteczna.
        if num % i == 0:
            factors.append(i)
            otherFactor = int(num / i)
            if otherFactor < MAX_KEY_LENGTH + 1 and otherFactor != 1:
                factors.append(otherFactor)
    return list(set(factors))  # Usunięcie wszelkich powtarzających się dzielników.


def getItemAtIndexOne(items):
    return items[1]


def getMostCommonFactors(seqFactors):
    # Przede wszystkim trzeba określić liczbę wystąpień dzielnika w słowniku seqFactors.
    factorCounts = {}  # Klucz jest dzielnikiem, a wartość określa liczbę jego wystąpień.

    # Klucze seqFactors są sekwencjami, natomiast wartości to listy dzielników
    # odstępów; seqFactors ma wartość taką jak {'GFD': [2, 3, 4, 6, 9, 12,
    # 18, 23, 36, 46, 69, 92, 138, 207], 'ALW': [2, 3, 4, 6, ...], ...}.
    for seq in seqFactors:
        factorList = seqFactors[seq]
        for factor in factorList:
            if factor not in factorCounts:
                factorCounts[factor] = 0
            factorCounts[factor] += 1

    # Następnie dzielnik i liczbę jego wystąpień należy umieścić w krotce.
    # Później tworzymy listę tych krotek, aby można było je sortować.
    factorsByCount = []
    for factor in factorCounts:
        # Wykluczenie dzielników większych niż wartość MAX_KEY_LENGTH.
        if factor <= MAX_KEY_LENGTH:
            # factorsByCount to lista krotek: (factor, factorCount).
            # factorsByCount ma wartości w postaci [(3, 497), (2, 487), ...].
            factorsByCount.append( (factor, factorCounts[factor]) )

    # Sortowanie listy według liczby wystąpień dzielnika.
    factorsByCount.sort(key=getItemAtIndexOne, reverse=True)

    return factorsByCount


def kasiskiExamination(ciphertext):
    # Wyszukanie sekwencji składających się z od 3 do 5 liter i wielokrotnie
    # występujących w szyfrogramie. repeatedSeqSpacings ma wartość podobną do
    # {'EXG': [192], 'NAF': [339, 972, 633], ... }.
    repeatedSeqSpacings = findRepeatSequencesSpacings(ciphertext)

    # (Omówienie seqFactors znajdziesz przy okazji omówienia funkcji getMostCommonFactors()).
    seqFactors = {}
    for seq in repeatedSeqSpacings:
        seqFactors[seq] = []
        for spacing in repeatedSeqSpacings[seq]:
            seqFactors[seq].extend(getUsefulFactors(spacing))

    # (Omówienie factorsByCount znajdziesz przy okazji omówienia funkcji getMostCommonFactors()).
    factorsByCount = getMostCommonFactors(seqFactors)

    # Wyodrębniamy z factorsByCount liczbę wystąpień dzielnika,
    # a następnie umieszczamy ją w allLikelyKeyLengths, aby łatwiej
    # było użyć tę wartość w dalszej części programu.
    allLikelyKeyLengths = []
    for twoIntTuple in factorsByCount:
        allLikelyKeyLengths.append(twoIntTuple[0])

    return allLikelyKeyLengths


def getNthSubkeysLetters(nth, keyLength, message):
    # Zwraca każdą co n-tą literę każdego zbioru keyLength liter w tekście,
    # dlatego: getNthSubkeysLetters(1, 3, 'ABCABCABC') zwraca 'AAA'
    #             getNthSubkeysLetters(2, 3, 'ABCABCABC') zwraca 'BBB'
    #             getNthSubkeysLetters(3, 3, 'ABCABCABC') zwraca 'CCC'
    #             getNthSubkeysLetters(1, 5, 'ABCDEFGHI') zwraca 'AF'

    # Użycie wyrażenia regularnego do usunięcia z message znaków innych niż litery.
    message = NONLETTERS_PATTERN.sub('', message)

    i = nth - 1
    letters = []
    while i < len(message):
        letters.append(message[i])
        i += keyLength
    return ''.join(letters)


def attemptHackWithKeyLength(ciphertext, mostLikelyKeyLength):
    # Próba ustalenia poszczególnych liter klucza.
    ciphertextUp = ciphertext.upper()
    # allFreqScores to lista składająca się z mostLikelyKeyLength list wewnętrznych.
    # Wspomniane listy wewnętrzne są listami freqScores.
    allFreqScores = []
    for nth in range(1, mostLikelyKeyLength + 1):
        nthLetters = getNthSubkeysLetters(nth, mostLikelyKeyLength, ciphertextUp)

        # freqScores to lista krotek następującego typu:
        # [(<litera>, <wynik dopasowania częstotliwości w języku angielskim>), ... ]
        # Lista jest sortowana według wyniku dopasowania częstotliwości, wyższy wynik oznacza lepsze dopasowanie.
        # Zapoznaj się z komentarzami w funkcji englishFreqMatchScore() w module freqAnalysis.py.
        freqScores = []
        for possibleKey in LETTERS:
            decryptedText = vigenereCipher.decryptMessage(possibleKey, nthLetters)
            keyAndFreqMatchTuple = (possibleKey, freqAnalysis.englishFreqMatchScore(decryptedText))
            freqScores.append(keyAndFreqMatchTuple)
        # Sortowanie według wyniku dopasowania częstotliwości.
        freqScores.sort(key=getItemAtIndexOne, reverse=True)

        allFreqScores.append(freqScores[:NUM_MOST_FREQ_LETTERS])

    if not SILENT_MODE:
        for i in range(len(allFreqScores)):
            # Użyj wyrażenia i + 1, aby pierwsza litera nie była nazwana zerową.
            print('Potencjalne litery dla znaku nr %s klucza: ' % (i + 1), end='')
            for freqScore in allFreqScores[i]:
                print('%s ' % freqScore[0], end='')
            print()  # Wyświetlenie znaku nowego wiersza.

    # Wypróbowanie każdej kombinacji najbardziej prawdopodobnych
    # liter dla każdego położenia w kluczu.
    for indexes in itertools.product(range(NUM_MOST_FREQ_LETTERS), repeat=mostLikelyKeyLength):
        # Utworzenie potencjalnego klucza na podstawie liter w allFreqScores.
        possibleKey = ''
        for i in range(mostLikelyKeyLength):
            possibleKey += allFreqScores[i][indexes[i]][0]

        if not SILENT_MODE:
            print('Sprawdzanie klucza: %s' % (possibleKey))

        decryptedText = vigenereCipher.decryptMessage(possibleKey, ciphertextUp)

        if detectEnglish.isEnglish(decryptedText):
            # Określenie wielkości liter zgodnej z wielkością w pierwotnej wiadomości.
            origCase = []
            for i in range(len(ciphertext)):
                if ciphertext[i].isupper():
                    origCase.append(decryptedText[i].upper())
                else:
                    origCase.append(decryptedText[i].lower())
            decryptedText = ''.join(origCase)

            # Użytkownik powinien potwierdzić, czy tekst został deszyfrowany prawidłowo.
            print('Potencjalnie udane złamanie szyfru za pomocą klucza %s:' % (possibleKey))
            print(decryptedText[:200])  # Wyświetlenie jedynie pierwszych 200 znaków.
            print()
            print('Wpisz D, aby zakończyć. Dowolny inny klawisz kontynuuje łamanie szyfru:')
            response = input('> ')

            if response.strip().upper().startswith('D'):
                return decryptedText

    # Jeżeli nie udało się złamać szyfru, wartością zwrotną funkcji jest None.
    return None


def hackVigenere(ciphertext):
    # Przede wszystkim trzeba użyć metody Kasiskiego w celu ustalenia
    # długości klucza zastosowanego do szyfrowania wiadomości.
    allLikelyKeyLengths = kasiskiExamination(ciphertext)
    if not SILENT_MODE:
        keyLengthStr = ''
        for keyLength in allLikelyKeyLengths:
            keyLengthStr += '%s ' % (keyLength)
        print('Wynik działania metody Kasiskiego wskazuje, że najbardziej prawdopodobne długości klucza to: ' + keyLengthStr + '\n')
    hackedMessage = None
    for keyLength in allLikelyKeyLengths:
        if not SILENT_MODE:
            print('Próba złamania szyfru za pomocą klucza o długości %s (%s potencjalnych kluczy)...' % (keyLength, NUM_MOST_FREQ_LETTERS ** keyLength))
        hackedMessage = attemptHackWithKeyLength(ciphertext, keyLength)
        if hackedMessage != None:
            break

    # Jeżeli żadna z długości klucza wskazanych przez metodę Kasiskiego się nie sprawdziła,
    # wówczas należy rozpocząć atak typu brute-force na pozostałe długości klucza.
    if hackedMessage == None:
        if not SILENT_MODE:
            print('Nie udało się złamać szyfru za pomocą prawdopodobnych długości kluczy. Rozpoczyna się atak typu brute-force na pozostałe długości kluczy...')
        for keyLength in range(1, MAX_KEY_LENGTH + 1):
            # Nie sprawdzamy długości kluczy wskazanych przez metodę Kasiskiego.
            if keyLength not in allLikelyKeyLengths:
                if not SILENT_MODE:
                    print('Próba złamania szyfru za pomocą klucza o długości %s (%s potencjalnych kluczy)...' % (keyLength, NUM_MOST_FREQ_LETTERS ** keyLength))
                hackedMessage = attemptHackWithKeyLength(ciphertext, keyLength)
                if hackedMessage != None:
                    break
    return hackedMessage


# Jeżeli program vigenereHacker.py został uruchomiony (a nie zaimportowany
# jako moduł), wówczas należy wywołać funkcję main().
if __name__ == '__main__':
    main()
